;slavst2a.asm to develop MSSP communications with the LCD PIC.
;john waller 2002nov17.

#DEFINE BANK0   BCF $03,5   ;STATUS bit 5
#DEFINE BANK1   BSF $03,5   ;STATUS bit 5

INDF:    .EQU $00	;Indirect address, bank 0, 1, 2, 3.
TMRO:	 .EQU $01	;Timer 0 value, bank 0, 3.
OPTION:  .EQU $01	;Option register, bank 1, 3.
PCL:     .EQU $02	;Program counter low byte, bank 0, 1, 2, 3.
STATUS:  .EQU $03	;Status flags, bank 0, 1, 2, 3.
FSR:     .EQU $04	;Indirect address pointer, bank 0, 1, 2, 3.
PORTA:   .EQU $05	;Port A, bank 0.
TRISA:   .EQU $05	;Port A direction register, bank 1.
PORTB:   .EQU $06	;Port B, bank 0, 2.
TRISB:   .EQU $06	;Port B direction register, bank 1, 3.
PORTC:   .EQU $07	;Port C, bank 0.
TRISC:   .EQU $07	;Port C direction register, bank 1.
PORTD:   .EQU $08	;Port D, bank 0.
TRISD:   .EQU $08	;Port D direction register, bank 1.
PORTE:   .EQU $09	;Port E, bank 0.
TRISE:   .EQU $09	;Port E direction register, bank 1.
PCLATH:	 .EQU $0A	;Program counter high byte setting register, bank 0, 1, 2, 3.
INTCON:  .EQU $0B	;Interrupt control, bank 0, 1, 2, 3.
PIR1:	 .EQU $0C	;Peripheral interrupt flags, bank 0.
PIE1:    .EQU $0C	;Peripheral interrupt enable, bank 1.
PIR2:    .EQU $0D	;Peripheral interrupt flags, bank 0.
PIE2:    .EQU $0D	;Peripheral interrupt enable, bank 1.
TMR1L:	 .EQU $0E	;Timer 1 low byte, bank 0.
TMR1H:	 .EQU $0F	;Timer 1 high byte, bank 0.
T1CON:	 .EQU $10	;Timer 1 control, bank 0.
SSPCON2: .EQU $11	;SSP configuration, bank 1.
SSPBUF:	 .EQU $13	;SSP address and data buffer, bank 0.
SSPADD:  .EQU $13	;SSP master baud rate/slave address, bank 1.
SSPCON:  .EQU $14	;SSP configuration, bank 0.
SSPSTAT: .EQU $14	;SSP status, bank 1.
ADCON1:	 .EQU $1F	;Port A A/D control register, bank 1.

;........GENERAL PURPOSE REGISTERS

SBYTCTR: .EQU $20	;SSP slave byte count, bank 0.
SBYTNUM: .EQU $21	;SSP number of data bytes to exchange, bank 0.
SINTCTR: .EQU $22	;SSP slave interrupt count, bank 0.
SSPTEMP: .EQU $20	;SSP temporary store, bank 1.
SYSTEM:  .EQU $23	;System parameters: bit 0 MSSP; 0 = send, 1 = read:
			;bit 1 MSSP 0 message not sent 1 sent:
			;others not yet assigned.
TMR1HS:	 .EQU $24	;Timer 1 high byte stored, bank 0.
ACTIONC: .EQU $25	;Action counter updated by button press, bank 0.
PORTAS:  .EQU $26	;Port A value stored, bank 0.

			;Registers to save working, FSR, STATUS, and PCLATH at interrupt.
			;Must be visible across all banks, 70H..7FH.
FSRINT:	.EQU $7C	;For FSR register, alias, FCH, 17CH, & 1FCH.
PCLINT:	.EQU $7D	;For PCLATH register, alias, FDH, 17DH, & 1FDH.
STAINT:	.EQU $7E	;For STATUS register, alias, FEH, 17EH, & 1FEH.
WRKINT:	.EQU $7F	;For working register, alias FFH, 17FH, & 1FFH.

			;Arrays accessible with IRP cleared.
			;A0h to A7h reserved for bank 1, non-array general purpose
			;registers.
MREABYT: .EQU $A8	;MSSP bytes read from slave to AFh.
MSENBYT: .EQU $B0	;MSSP bytes sent to slave to B7h.

;........CONSTANT VALUES

ERRINT:	.EQU $20	;Tag for an interrupt error.
ERRSAD:	.EQU $30	;Tag for a slave address error.
ERRSDA:	.EQU $40	;Tag for a slave data error.
MISSPB:	.EQU %00101101	;Mask in important bits in SSPSTAT.
MSSPRAM: .EQU %00101101	;SSP read address status mask.
MSSPRAC: .EQU %00001100	;SSP read address comparison.
MSSPRDM: .EQU %00101001	;SSP read data status mask.
MSSPRDC: .EQU %00101000	;SSP read data comparison.
MSSPSAM: .EQU %00101101	;SSP send address status mask.
MSSPSAC: .EQU %00001001	;SSP send address comparison.
MSSPSDM: .EQU %00101101	;SSP send data status mask.
MSSPSDC: .EQU %00101001	;SSP send data comparison.
SLVADD: .EQU $10	;Slave address; upper 7 bits; equate must be even;
			;do not use F8h, FAh, FCh, FEh, or zero.
			;16F877 manual says only 112 valid addresses available;
			;don't know what the other invalid ones are.
SSPWAB:	.EQU %00001001	;SSP write address bits.
SSPWDB:	.EQU %00101001	;SSP write data bits.

;........BIT VALUES

ACKDT:  .EQU 5		;SSP master acknowledge data.
ACKEN:  .EQU 4		;SSP master 
ACKSTAT:.EQU 6		;SSP master acknowledge status.
BCLIE   .EQU 3		;Bus collision interrupt enable.
BCLIF:  .EQU 3		;Bus collision interrupt flag.
BF:     .EQU 0		;SSP buffer full.
C:      .EQU 0          ;Carry status.
CKE:    .EQU 6		;Clock edge select.
CKP:	.EQU 4		;Hold/enable SSP clock pulse by slave.
DIR:	.EQU 0		;MSSP direction; 0 send, 1 read.
DNA     .EQU 5		;Last byte received or transmitted was data or address.
F:      .EQU 1          ;Select file register.
GCEN:   .EQU 7		;SSP general call enable.
GIE:    .EQU 7		;General interrupt enable.
P:      .EQU 4		;Stop bit.
PEIE:   .EQU 6		;Peripheral interrupt enable.
PEN:    .EQU 2		;SSP master stop enable.
RCEN:   .EQU 3		;SSP receive enable.
RNW:    .EQU 2		;Read/write information following the last address match.
RP0:    .EQU 5          ;STATUS bank control bit.
RP1:    .EQU 6          ;STATUS bank control bit.
RSEN:	.EQU 1		;SSP master repeat start enable.
SEN:    .EQU 0		;SSP master start enable.
SMP:    .EQU 7		;Slew rate control.
SSPIE:  .EQU 3		;SSP interrupt enable.
SSPIF:  .EQU 3		;SSP interrupt flag.
SSPEN:  .EQU 5		;Configure SDA and SCL pins for SSP.
SSPOV:  .EQU 6		;SSP receive overflow detect.
UA:     .EQU 1		;Update address; 10-bit address mode only.
W:      .EQU 0          ;Select working register.
WCOL:   .EQU 7		;SSP write collision detect.
Z:      .EQU 2          ;Zero status.

;..........

        .ORG $0004
	goto INTERRUPT
        .ORG $0005

	goto INITIALISE	;jump over sub-page 0 tables to INITIALISE.

		;page 0, sub-page 0 tables: make sure they don't overrun.
		
;..........

BACKGROUND:	;Main part of program in background; entered from INITIALISE.
	goto BACKGROUND		;Loop back.

;..........

CHECKANDSTOP:	;Writes selected registers to ports and pauses.
		;Called from ERRORINT, ERRORSADDR, ERRORSDATR; calls, none.
	BANK0			;
	andlw %11100000		;Write high bits in W to port C
	movwf PORTC		;to indicate where in program stop occurred.
	bsf PORTE,2		;Signal program has reached here.
	btfss PIR1,SSPIF	;Show SSP
	goto CHAST1		;interrupt
	bsf PORTC,2		;bit
	goto CHAST2		;on
CHAST1:	bcf PORTC,2		;port C,2.
CHAST2:	BANK1			;Write
	movf SSPSTAT,W		;register
	BANK0			;contents
	movwf PORTB		;to port
	movf SSPCON,W		;Write register
	movwf PORTD		;contents to port.
CHAST3:	goto CHAST3		;Loop indefinitely.

;..........

DELAY0PT2S:	;Delays approximately 0.2 second using timer 1.
		;Called from ...............; calls none.
	movf TMR1H,W		;Store current value of
	movwf TMR1HS		;timer 1 high byte.
DE0P21:	movf TMR1H,W		;Skip if timer
	subwf TMR1HS,W		;high byte
	sublw 1			;is one less than
	btfss STATUS,Z		;stored value.
	goto DE0P21		;Otherwise continue to wait.
	return			;

;..........

DISPACTIONCTR:	;Displays the action counter, showing lower three bits on port
		;A,0 ,1, 2.
	movf ACTIONC,W		;Mask in lower three bits
	andlw %00000111		;of action counter
	movwf PORTAS		;and store result.
	movf PORTA,W		;Mask in upper
	andlw %00111000		;three bits of port A,
	addwf PORTAS,W		;add in lower three bits and
	movwf PORTA		;write to port A.
	movf ACTIONC,W		;Carry out action
	call SWITCHCASE8	;designated by value.
	return

;..........

ERRORINT:	;Sets flags for an interrupt error and halts execution.
		;Called from INTERRUPT; calls, CHECKANDSTOP.
	movlw ERRINT		;Set interrupt error tag, write SSPSTAT and SSPCON to
	call CHECKANDSTOP	;ports B and D, respectively, and halt.

;..........

ERRORSADDR:	;Sets flags for a slave read address error and halts execution.
		;Called from HANDSSPADDR; calls, CHECKANDSTOP.
	movlw ERRSAD		;Set slave read address error tag, write SSPSTAT and
				;SSPCON to ports B and D,
	call CHECKANDSTOP	;respectively, and halt.

;..........

ERRORSDATR:	;Sets flags for a slave read data error and halts execution.
		;Called from HANDSSPDATR; calls, CHECKANDSTOP.
	movlw ERRSDA		;Set slave read data error tag, write SSPSTAT and
				;SSPCON to ports B and D,
	call CHECKANDSTOP	;respectively, and halt.

;..........

HANDSSPMRAD:	;Sets system to master read, checks for address errors, and reads
		;buffer to empty it.
		;Called from INTERRUPT; calls none.
	bsf SYSTEM,DIR		;Put system in master read.
	BANK1			;
	movf SSPSTAT,W		;Get SSP status register and mask
	andlw MSSPRAM		;in important flags, putting
	movwf SSPTEMP		;result in temporary store.
	movlw MSSPRAC		;If the flags do
	xorwf SSPTEMP,W		;not comply with the
	btfss STATUS,Z		;required pattern
	call ERRORSADDR		;report error and stop.
	BANK0			;
	btfsc SSPCON,SSPOV	;Skip if no overflow, otherwise
	call ERRORSADDR		;report error and stop.
	movf SSPBUF,W		;Otherwise do a dummy read of the buffer to empty it.
	return			;
	
;..........

HANDSSPMRCF:	;Checks for master read errors.
		;Called from INTERRUPT; calls none.
	BANK1			;
	movf SSPSTAT,W		;Get SSP status register and mask
	andlw MSSPRDM		;in important flags, putting
	movwf SSPTEMP		;result in temporary store.
	movlw MSSPRDC		;If the flags do
	xorwf SSPTEMP,W		;not comply with the
	btfss STATUS,Z		;required pattern
	call ERRORSDATR		;report error and stop.
	BANK0			;
	btfsc SSPCON,SSPOV	;Skip if no overflow, otherwise
	call ERRORSDATR		;report error and stop.
	return			;

;..........

HANDSSPMRDA:	;Sets data byte for master to read, and releases clock.
		;Called from INTERRUPT; calls none.
	movlw MREABYT		;Point to read byte
	addwf SBYTCTR,W		;array current
	movwf FSR		;storage location.
	movf INDF,W		;Put byte
	movwf SSPBUF		;in buffer.
	bsf SSPCON,CKP		;Release clock to allow transmission.
	incf SBYTCTR,F		;Increment byte counter.
	incf SINTCTR,F		;Increment interrupt counter.
	return			;

;..........

HANDSSPMSAD:	;Sets system to master send, checks for address errors, and reads
		;buffer to empty it.
		;Called from INTERRUPT; calls none.
	bcf SYSTEM,DIR		;Put system in master send.
	BANK1			;
	movf SSPSTAT,W		;Get SSP status register and mask
	andlw MSSPSAM		;in important flags, putting
	movwf SSPTEMP		;result in temporary store.
	movlw MSSPSAC		;If the flags do
	xorwf SSPTEMP,W		;not comply with the
	btfss STATUS,Z		;required pattern
	call ERRORSDATR		;report error and stop.
	BANK0			;
	btfsc SSPCON,SSPOV	;Skip if no overflow, otherwise
	call ERRORSDATR		;report error and stop.
	movf SSPBUF,W		;Otherwise do a dummy read of the buffer to empty it.
	incf SINTCTR,F		;Increment interrupt counter.
	return			;

;..........

HANDSSPMSDA:	;Gets data byte sent by master and stores it.
		;Called from INTERRUPT; calls none.
	BANK1			;
	movf SSPSTAT,W		;Get SSP status register and mask
	andlw MSSPSDM		;in important flags, putting
	movwf SSPTEMP		;result in temporary store.
	movlw MSSPSDC		;If the flags do
	xorwf SSPTEMP,W		;not comply with the
	btfss STATUS,Z		;required pattern
	call ERRORSDATR		;report error and stop.
	BANK0			;
	btfsc SSPCON,SSPOV	;Skip if no overflow, otherwise
	call ERRORSDATR		;report error and stop.
	movlw MSENBYT		;Point to send byte
	addwf SBYTCTR,W		;array current
	movwf FSR		;storage location.	
	movf SSPBUF,W		;Put buffer
	movwf INDF		;contents there.
	incf SBYTCTR,F		;Increment byte and
	incf SINTCTR,F		;interrupt counters.
	return			;

;..........

INITI2CSLVE:	;Initialises the I2C slave device, and interrupts for it.
		;Called from INITIALISE; calls none.
	movlw %00110110		;Bits 7, 6, flags; bit 5 enable SDA and SCL port pins;
	movwf SSPCON		;bit 4, clock release control;
				;bits 3..0, I2C 7-bit address slave mode.
	BANK1			;
	movlw %10000000		;Bit 7, slew rate control disabled; bit 6, comply with
	movwf SSPSTAT		;I2C protocol; bits 5..0, flags.
	clrf SSPCON2		;All bits flags.
	movlw SLVADD		;Set slave address
	movwf SSPADD		;for this module.
	bsf PIE1,SSPIE		;Enable SSP interrupt.
	BANK0			;
	bsf INTCON,PEIE		;Enable peripheral interrupts.
	bsf INTCON,GIE		;Enable general interrupts.
	return			;
	
;..........

INITIALISE:	;Initialising ports and others. Entered at startup.
		;Called from none; calls INITI2CSLVE.
	clrf PCLATH		;Page and sub-page 0.
	clrf STATUS		;Bank 0 and clear all flags.
				;Clear all gpr locations in banks 0 and 1.
	movlw $20		;First gpr address in bank 0
	movwf FSR		;to indirect address register.
INITB0:	clrf INDF		;Clear location pointed to.
	incf FSR,F		;Select next address.
	btfss FSR,7		;Skip if end of bank 0.
	goto INITB0		;
	movlw $A0		;First gpr address in bank 1
	movwf FSR		;to indirect address register.
INITB1:	clrf INDF		;Clear location pointed to.
	incf FSR,F		;Select next address.
	btfss STATUS,Z		;Skip if end of bank 1.
	goto INITB1		;
				;Set up ports and timers.
	clrf PORTA		;Clear port.
        clrf PORTB		;Clear port.
	clrf PORTC		;Clear port.
        clrf PORTD		;Clear port.
	clrf PORTE		;Clear port.
        BANK1			;Select bank 1.
	movlw %00000110		;Port A as digital
	movwf ADCON1		;input/output.
	movlw %00110000		;Port A0..A3 outputs, A4, A5 inputs.
        movwf TRISA     	;as input.
        clrf TRISB      	;Port B0-B7 as output.
	movlw %00011000		;Port C all bits output, except
	movwf TRISC		;3 and 4 must be input for I2C.
	movlw %00000000		;Port D bits all
	movwf TRISD		;outputs.
	movlw %00000000		;Port E all bits as output,
	movwf TRISE		;disable slave port.
        BANK0			
        clrf INTCON		;No timers, init slave sets interrupt bits.
        clrf INTCON		;No interrupts or timers yet.
	movlw %00110001		;Set up timer 1; bit 0 enable, bit 1 Fosc/4, bit 2 X,
	movwf T1CON		;bit 3 not internal oscillator, bits 4/5 prescale 8,
				;bits 6/7 not implemented.
        call INITI2CSLVE	;Initialise I2C slave mode.
        movlw 4			;Set number of data bytes
        movwf SBYTNUM		;to four.
        call SETDUMBYT2READ	;Set dummy bytes for master to read.
        call TESTACTIONCTR	;Loops here forever.
        goto BACKGROUND		;
        
;...........

INTERRUPT:	;Interrupt routine goes here.
		;Called from none; calls, ERRORINT, HANDSSPADDR, HANDSSPDATR.
				;Save registers and set up to execute interrupt.
	movwf WRKINT		;Save working register.
	swapf STATUS,W		;Save status
	movwf STAINT		;register, without affecting flags.
	movf PCLATH,W		;Save page
	movwf PCLINT		;register.
	movf FSR,W		;Save indirect
	movwf FSRINT		;address register.
	clrf STATUS		;Set bank, page,
	clrf PCLATH		;and sub-page 0.
	
				;Execute interrupt.
	btfss PIR1,SSPIF	;If not SSP interrupt, call error
	call ERRORINT		;routine and halt execution, otherwise service
	bcf PIR1,SSPIF		;interrupt, first clearing interrupt flag.
	movf SINTCTR,F		;Skip if byte
	btfss STATUS,Z		;count zero.
	goto INTRT3		;
				;Handle address section.
	BANK1			;
	btfss SSPSTAT,RNW	;Skip if master read.
	goto INTRT2		;
	BANK0			;
	call HANDSSPMRAD	;Handle master read address.
INTRT1:	BANK0			;
	call HANDSSPMRDA	;Set up for master data read.
	goto INTEND		;
				;
INTRT2:	BANK0			;
	call HANDSSPMSAD	;Handle master send address.
	goto INTEND		;
				;
INTRT3:	btfss SYSTEM,DIR	;Skip if master read set.
	goto INTRT4		;
	call HANDSSPMRCF	;Handle check master read flags.
	BANK1			;
	btfsc SSPSTAT,RNW	;Skip if master sent NACK.
	goto INTRT1		;Otherwise send another data byte.
	BANK0			;
	clrf SBYTCTR		;Clear both
	clrf SINTCTR		;counters.
	goto INTEND		;
				;	
INTRT4:	call HANDSSPMSDA	;Read and store data sent by master.
	movf SBYTNUM,W		;Skip if enough
	subwf SBYTCTR,W		;bytes have
	btfss STATUS,C		;been sent.
	goto INTEND		;
	clrf SBYTCTR		;Clear both
	clrf SINTCTR		;counters.
	goto INTEND		;
	
INTEND:				;Restore registers and return.
	movf FSRINT,W		;Restore indirect
	movwf FSR		;address register.
	movf PCLINT,W		;Restore page
	movwf PCLATH		;register.
	swapf STAINT,W		;Restore status
	movwf STATUS		;register.
	swapf WRKINT,F		;Restore working
	swapf WRKINT,W		;register.
	retfie			;

;..........

SETDUMBYT2READ:	;Sets up the read array with dummy data.
		;Called from .........., calls none.
	movlw MREABYT		;
	movwf FSR		;
	movlw %10111010		;
	movwf INDF		;
	incf FSR,F		;
	movlw %01101110		;
	movwf INDF		;
	incf FSR,F		;
	movlw %11111010		;
	movwf INDF		;
	incf FSR,F		;
	movlw %01101111		;
	movwf INDF		;
	return			;
	
;..........

SWITCHCASE8:	;Takes first 3 bits of working register and does an 8-way switch-
		;case selection.
		;Called from ..............; calls ......
	andlw %00000111 	;Mask in 3 lower bits.
        btfss STATUS,Z  	;
        goto SWCAS1     	;
        			;Value 0.
        movlw MSENBYT		;Point to array.
        movwf FSR		;Get value and
        movf INDF,W		;display on
        movwf PORTB		;port B.
        return     		;
SWCAS1: addlw $0FF      	;
        btfss STATUS,Z  	;
        goto SWCAS2     	;
        			;Value 1.
        movlw MSENBYT		;Point to array.
        addlw 1			;Point to member of interest.
        movwf FSR		;Get value and
        movf INDF,W		;display on
        movwf PORTB		;port B.
        return     		;
SWCAS2: addlw $0FF      	;
        btfss STATUS,Z  	;
        goto SWCAS3     	;
        			;Value 2.
        movlw MSENBYT		;Point to array.
        addlw 2			;Point to member of interest.
        movwf FSR		;Get value and
        movf INDF,W		;display on
        movwf PORTB		;port B.
        return     		;
SWCAS3: addlw $0FF      	;
        btfss STATUS,Z  	;
        goto SWCAS4     	;
        			;Value 3.
        movlw MSENBYT		;Point to array.
        addlw 3			;Point to member of interest.
        movwf FSR		;Get value and
        movf INDF,W		;display on
        movwf PORTB		;port B.
        return     		;
SWCAS4: addlw $0FF      	;
        btfss STATUS,Z  	;
        goto SWCAS5     	;
        			;Value 4.
        return     		;
SWCAS5: addlw $0FF      	;
        btfss STATUS,Z  	;
        goto SWCAS6     	;
        			;Value 5.
        return     		;
SWCAS6: addlw $0FF      	;
        btfss STATUS,Z  	;
        goto SWCAS7     	;
        			;Value 6.
        return     		;
        			;Value 7 and default.
SWCAS7: return             	;

;..........

TESTACTIONCTR:	;Tests the action counter.
	call DISPACTIONCTR	;Display low bits on port A,0,1,2.
	call WAITFORSWPRESS	;Increment
	incf ACTIONC,F		;counter.
	goto TESTACTIONCTR	;

;..........

WAITFORSWPRESS:	;Waits for a momentary-action switch to pull down port A,4
		;and release it, with 0.2 second de-bounce delays.
	btfsc PORTA,4		;Wait for button to
	goto WAITFORSWPRESS	;pull port pin down.
	call DELAY0PT2S		;De-bounce delay.
WTFSP1:	btfss PORTA,4		;Wait for button to
	goto WTFSP1		;release port pin.
	call DELAY0PT2S		;De-bounce delay.
	return			;
;..........

        .END
